#include "ArpVaccineP.h"

#include <stdio.h>
#include <stdlib.h>
#include <be/interface/CheckBox.h>
#include <be/interface/MenuItem.h>
#include "ArpKernel/ArpDebug.h"
#include "ArpLayout/ArpViewWrapper.h"
#include "ArpLayout/ViewStubs.h"
#include "ArpViewsPublic/ArpIntFormatterI.h"
#include "ArpViews/ArpIntControl.h"
#include "ArpViews/ArpKnobControl.h"
#include "AmPublic/AmControls.h"
#include "AmPublic/AmFilterConfigLayout.h"
#include "AmPublic/AmGlobalsI.h"
#include "AmPublic/AmMotionI.h"

ArpMOD();
static AmStaticResources gRes;

static const char*		FOLLOW_Y_STR			= "Follow Y";
static const char*		FOLLOW_X_STR			= "Follow X";
static const char*		INVERT_X_STR			= "Invert X";
static const char*		FREQUENCY_STR			= "frequency";
static const char*		AMOUNT_STR				= "amount";
static const char*		MOTION_FROM_TRACK_STR	= "Motion From Track";
static const char*		MOTION_STR				= "motion";

static const int32		MIN_AMOUNT				= -400;
static const int32		MAX_AMOUNT				= 400;

/**********************************************************************
 * _VACCINE-FILTER-SETTINGS and _TOOL-SETTINGS
 **********************************************************************/
class _VaccineFilterSettings : public AmFilterConfigLayout
{
public:
	_VaccineFilterSettings(	AmFilterHolderI* target,
							const BMessage& initSettings);

	virtual void AttachedToWindow();
	virtual void MessageReceived(BMessage* msg);

protected:
	typedef AmFilterConfigLayout inherited;
	BCheckBox*			mFollowXBox;
	BCheckBox*			mInvertXBox;
	ArpMenuField*		mRhythmField;
	BCheckBox*			mTrackMotionBox;
	AmMotionEditor*		mEditor;
	
	void AddViews(	ArpBaseLayout* toLayout, ArpConfigureImpl& impl,
					float labelW, float intW,
					const BMessage& initSettings);
};

/*****************************************************************************
 * ARP-VACCINE-FILTER
 *****************************************************************************/
ArpVaccinePFilter::ArpVaccinePFilter(	ArpVaccinePAddOn* addon,
										AmFilterHolderI* holder,
										const BMessage* settings)
		: AmFilterI(addon), mAddOn(addon), mHolder(holder), mMotion(NULL),
		  mChangeFlags(FOLLOW_Y_FLAG), mFrequency(100), mAmount(100)
{
	mSeed = int32(system_time()/100);
	InitMotion();
	if (settings) PutConfiguration(settings);
}

ArpVaccinePFilter::~ArpVaccinePFilter()
{
	delete mMotion;
}

AmEvent* ArpVaccinePFilter::HandleEvent(AmEvent* event, const am_filter_params* params)
{
	return HandleBatchEvents(event, params);
}

AmEvent* ArpVaccinePFilter::HandleBatchEvents(	AmEvent* event,
												const am_filter_params* params,
												const AmEvent* /*lookahead*/)
{
	if (!event || !mHolder || !params || !params->cur_signature) return event;
	AmMotionChange*		motionEvent = NULL;
	bool				deleteMotionEvent = false;
	if (mChangeFlags&MOTION_FROM_TRACK_FLAG) {
		motionEvent = params->MotionChange(mHolder->TrackId() );
	} else {
		if (mMotion) {
			motionEvent = new AmMotionChange(mMotion, 1, 0);
			deleteMotionEvent = true;
		}
	}

	if (!motionEvent) return event;
	VaccinateBatchEvents(event, params, motionEvent, NULL);
	
	if (deleteMotionEvent) motionEvent->Delete();
	return event;
}

AmEvent* ArpVaccinePFilter::HandleBatchToolEvents(	AmEvent* event,
													const am_filter_params* params,
													const am_tool_filter_params* toolParams,
													const AmEvent* /*lookahead*/)
{
	if (!event || !mHolder) return event;
	event->SetNextFilter(mHolder->FirstConnection() );
	if (!params || !params->cur_signature || !toolParams) return event;
	AmMotionChange*		motionEvent = NULL;
	bool				deleteMotionEvent = false;
	if (mChangeFlags&MOTION_FROM_TRACK_FLAG) {
		motionEvent = params->MotionChange(toolParams->track_context);
	} else {
		if (mMotion) {
			motionEvent = new AmMotionChange(mMotion, 1, 0);
			deleteMotionEvent = true;
		}
	}

	if (!motionEvent) return event;
	VaccinateBatchEvents(event, params, motionEvent, toolParams);
	
	if (deleteMotionEvent) motionEvent->Delete();
	return event;
}

void ArpVaccinePFilter::VaccinateBatchEvents(AmEvent* event, const am_filter_params* params,
											AmMotionChange* curMotion,
											const am_tool_filter_params* toolParams)
{
	ArpASSERT(params && params->cur_signature);
	if (!curMotion) return;
	AmSignature		measure(*(params->cur_signature) );
	int32			motionMeasureOffset = 0;

	while (event) { 
		float		y;
		if ( (event->Type() == event->NOTEON_TYPE || event->Type() == event->NOTEOFF_TYPE)
				&& ShouldVaccinate()
				&& am_motion_hit_at_time(	event->StartTime(), &curMotion, &motionMeasureOffset,
										measure, &y) == B_OK) {
//			printf("Time %lld hit is %f\n", event->StartTime(), y);
			float		amount = (mAmount / 100) * y;
			float		yAmount = 1;
			if (toolParams) {
				if (mChangeFlags&FOLLOW_Y_FLAG) {
					float	pixels = (float)(toolParams->orig_y_pixel - toolParams->cur_y_pixel) / 100;
					amount = pixels;
					yAmount = y;
				}
				if (mChangeFlags&FOLLOW_X_FLAG) {
					AmTime	s = toolParams->start_time, e = toolParams->end_time, c = toolParams->cur_time;
					if (s >= 0 && e >= 0 && c >= 0) {
						float		xamt = am_x_amount(toolParams, event->StartTime() );
						if (mChangeFlags&INVERT_X_FLAG) xamt = fabs(1 -xamt);
						amount = amount * xamt;
					}
				}
			}

			if (event->Type() == event->NOTEON_TYPE) {
				AmNoteOn*	e = dynamic_cast<AmNoteOn*>(event);
				if (e) {
					int32		delta = int32(yAmount * amount * 127);
					int32		newNote = (int32)(e->Note() + delta);
					if (newNote > 127) newNote = 127;
					else if (newNote < 0) newNote = 0;
					e->SetNote(newNote);
				}
			} else if (event->Type() == event->NOTEOFF_TYPE) {
				AmNoteOff*	e = dynamic_cast<AmNoteOff*>(event);
				if (e) {
					int32		delta = int32(yAmount * amount * 127);
					int32		newNote = (int32)(e->Note() + delta);
					if (newNote > 127) newNote = 127;
					else if (newNote < 0) newNote = 0;
					e->SetNote(newNote);
				}
			}

		}
		event->SetNextFilter(mHolder->FirstConnection() );
		event = event->NextEvent();
	}

	measure.RemoveEvent();
}

#if 0
	/* Get how much of my rhythm I should mix in, from -400% to 400% (i.e. an amount
	 * of -4.0 to 4.0.  If I have tool params, get the amount based on the current
	 * mouse Y position.
	 */
	float		hit = hit_amount(event->StartTime(), mMotion, params->cur_signature);
	float		amount = (mAmount / 100) * hit;
	float		yAmount = 1;
	if (mChangeFlags&FOLLOW_Y_FLAG) {
		float	pixels = (float)(toolParams->orig_y_pixel - toolParams->cur_y_pixel) / 100;
		amount = pixels;
		yAmount = hit;
	}
	if (mChangeFlags&FOLLOW_X_FLAG) {
		AmTime	s = toolParams->start_time, e = toolParams->end_time, c = toolParams->cur_time;
		if (s >= 0 && e >= 0 && c >= 0) {
			amount = amount * am_x_amount(toolParams, event->StartTime() );
		}
	}

	if (amount > 1) amount = 1;
	else if (amount < -1) amount = -1;

	if (event->Type() == event->NOTEON_TYPE) {
		AmNoteOn*	e = dynamic_cast<AmNoteOn*>(event);
		if (e && mChangeFlags&CHANGE_VELOCITY_FLAG) {
			int32	newVel = (int32)(e->Velocity() + (amount * yAmount * 127));
			if (newVel > 127) newVel = 127;
			else if (newVel < 0) {
				e->Delete();
				return NULL;
			}
			e->SetVelocity(newVel);
		}
		if (e && mChangeFlags&CHANGE_PITCH_FLAG) {
			int32		delta = 0;
			int32	max = abs(toolParams->orig_y_value - toolParams->cur_y_value);
			delta = int32(yAmount * 128);
			if (delta < 0) {
				max = 0 - max;
				if (delta < max) delta = max;
			} else {
				if (delta > max) delta = max;
			}
			int32	newNote = (int32)(e->Note() + delta);
			if (newNote > 127) newNote = 127;
			else if (newNote < 0) newNote = 0;
			e->SetNote(newNote);
		}
	} else if (event->Type() == event->NOTEOFF_TYPE) {
		AmNoteOff*	e = dynamic_cast<AmNoteOff*>(event);
		if (e && mChangeFlags&CHANGE_PITCH_FLAG) {
			int32	newNote = (int32)(e->Note() + (amount * yAmount * 127));
			if (newNote > 127) newNote = 127;
			else if (newNote < 0) newNote = 0;
			e->SetNote(newNote);
		}
	}

	return event;
}
#endif


status_t ArpVaccinePFilter::GetConfiguration(BMessage* values) const
{
	status_t err = AmFilterI::GetConfiguration(values);
	if (err != B_OK) return err;

	if (values->AddBool(FOLLOW_Y_STR, mChangeFlags&FOLLOW_Y_FLAG) != B_OK) return B_ERROR;
	if (values->AddBool(FOLLOW_X_STR, mChangeFlags&FOLLOW_X_FLAG) != B_OK) return B_ERROR;
	if (values->AddBool(INVERT_X_STR, mChangeFlags&INVERT_X_FLAG) != B_OK) return B_ERROR;
	if (values->AddBool(MOTION_FROM_TRACK_STR, mChangeFlags&MOTION_FROM_TRACK_FLAG) != B_OK) return B_ERROR;
	if (values->AddInt32(FREQUENCY_STR, mFrequency) != B_OK) return B_ERROR;
	if (values->AddInt32(AMOUNT_STR, mAmount) != B_OK) return B_ERROR;
	if (mMotion) {
		BMessage	msg;
		if ((err = mMotion->WriteTo(msg)) != B_OK) return err;
		if ((err = values->AddMessage(MOTION_STR, &msg)) != B_OK) return err;
	}

	return B_OK;
}

status_t ArpVaccinePFilter::PutConfiguration(const BMessage* values)
{
	status_t err = AmFilterI::PutConfiguration(values);
	// Want to make sure that batch mode is always turned on.
	SetFlag(BATCH_FLAG, true);

	if (err != B_OK) return err;
	bool		b;
	if (values->FindBool(FOLLOW_Y_STR, &b) == B_OK) {
		if (b) mChangeFlags |= FOLLOW_Y_FLAG;
		else mChangeFlags &= ~FOLLOW_Y_FLAG;
	}
	if (values->FindBool(FOLLOW_X_STR, &b) == B_OK) {
		if (b) mChangeFlags |= FOLLOW_X_FLAG;
		else mChangeFlags &= ~FOLLOW_X_FLAG;
	}
	if (values->FindBool(INVERT_X_STR, &b) == B_OK) {
		if (b) mChangeFlags |= INVERT_X_FLAG;
		else mChangeFlags &= ~INVERT_X_FLAG;
	}
	if (values->FindBool(MOTION_FROM_TRACK_STR, &b) == B_OK) {
		if (b) mChangeFlags |= MOTION_FROM_TRACK_FLAG;
		else mChangeFlags &= ~MOTION_FROM_TRACK_FLAG;
	}

	BMessage	msg;
	if (values->FindMessage(MOTION_STR, &msg) == B_OK) {
		delete mMotion;
		mMotion = AmMotionI::NewMotion(msg);
	}

	int32		i;
	if (values->FindInt32(FREQUENCY_STR, &i) == B_OK) mFrequency = i;
	if (values->FindInt32(AMOUNT_STR, &i) == B_OK) mAmount = i;

	return B_OK;
}

status_t ArpVaccinePFilter::Configure(ArpVectorI<BView*>& panels)
{
	BMessage config;
	status_t err = GetConfiguration(&config);
	if (err != B_OK) return err;
	panels.push_back(new _VaccineFilterSettings(mHolder, config));
	return B_OK;
}

void ArpVaccinePFilter::Start(uint32 context)
{
	if (context&TOOL_CONTEXT) mSeed = int32(system_time()/100);
}

bool ArpVaccinePFilter::ShouldVaccinate() const
{
	if (mFrequency <= 0) return false;
	if (mFrequency >= 100) return true;

	srand( (int32)(system_time()/100) );
	int32 percent = rand() % 100;

	if (percent < mFrequency) return true;
	else return false;
}

void ArpVaccinePFilter::InitMotion()
{
	delete mMotion;
	mMotion = NULL;
	/* See if the standard Progress Up motion is installed.  If so,
	 * initialize to that.
	 */
	mMotion = AmGlobals().NewMotion("Progress Up");
	if (mMotion) return;
	/* Otherwise, initialize myself to the first installed motion.
	 */
	BString		label, key;
	if (AmGlobals().GetMotionInfo(0, label, key) != B_OK) return;
	mMotion = AmGlobals().NewMotion(key);
	if (mMotion) return;
	/* Finally, if there are none, just a blank motion.
	 */
	BMessage	config;
	mMotion = AmMotionI::NewMotion(config);
}

/*****************************************************************************
 * ARP-VACCINE-V-ADD-ON
 *****************************************************************************/
void ArpVaccinePAddOn::LongDescription(BString& name, BString& str) const
{
	AmFilterAddOn::LongDescription(name, str);
	str << "<p>I alter the pitch of notes based on my motion.  For each note
		I receive, I check where it falls on the motion, then alter the pitch
		accordingly.</p>
		
		<h4>Amount</h4>
			<p>Amount determines how much the current motion value affects the pitch.
			When the Amount: value is at 100%, then the final pitch will exactly
			match the current motion value.  For example, if the note being processed
			occurs at a motion hit that is five steps above the 0 center line, then
			the note will be increased by five steps.</p>
			
			<p>Tools follow Y is only meaningful when this filter is being used from
			a tool.  It causes the Amount value to be set by how far the mouse has
			traveled from the original point.  If the mouse travels below the original
			point, the motion becomes inverted.</p>
			
			<p>Tools follow X causes the Amount to taper off towards either end of
			the selected events.</p>
			
			<p>Invert X is only valid if Tools follow X is on.  It inverts the tapering
			effect, essentially turning it from a cone to a bowl.</p>

		<h4>Frequency</h4>
			<p>This parameter determines how frequently notes will be processed.  When
			set to 0% (Never) this filter is effectively bypassed.  When set to 100%
			(Always) every note is processed.</p>
	
		<h4>Motion Selection</h4>
			<p>The Use Motion button presents a list of all the motions currently
			installed in the system.  Selecting one will copy it into the motion editor
			at the bottom of the window.</p>
			
			<p>If Use motion from track is on, then my motion is ignored.  Instead,
			I use whatever the current track motion is active for each event being
			processed.  If no track motion is set, then nothing happens.</p>";
}

void ArpVaccinePAddOn::GetVersion(int32* major, int32* minor) const
{
	*major = 1;
	*minor = 0;
}

BBitmap* ArpVaccinePAddOn::Image(BPoint requestedSize) const
{
	const BBitmap* bm = gRes.Resources().FindBitmap("Class Icon");
	if (bm) return new BBitmap(bm);
	return NULL;
}

extern "C" _EXPORT AmFilterAddOn* make_nth_filter(int32 n, image_id /*you*/,
												  const void* cookie, uint32 /*flags*/, ...)
{
	if (n == 0) return new ArpVaccinePAddOn(cookie);
	return NULL;
}

// #pragma mark -

/*************************************************************************
 * _OFF-PERCENT-FORMAT
 * A class that formats 0 to off and everything else to a percent.
 *************************************************************************/
class _OffPercentFormat : public ArpIntFormatterI
{
public:
	_OffPercentFormat()	{ ; }

	virtual void FormatInt(int32 number, BString& out) const
	{
		if (number == 0) out << "Off";
		else out << number << '%';
	}
};

/**********************************************************************
 * _VACCINE-FILTER-SETTINGS
 **********************************************************************/
static const uint32		MOTION_MSG		= 'iRtm';
static const char*		MOTION_KEY_STR	= "motion_key";

static ArpMenuField* new_rhythm_menu_field()
{
	BMenu*		menu = new BMenu("Use Motion");
	if (!menu) return NULL;
	BString		label, key;
	for (uint32 k = 0; AmGlobals().GetMotionInfo(k, label, key) == B_OK; k++) {
		BMessage*		msg = new BMessage(MOTION_MSG);
		if (msg) {
			msg->AddString(MOTION_KEY_STR, key);
			BMenuItem*	item = new BMenuItem(label.String(), msg);
			if (!item) delete msg;
			else menu->AddItem(item);
		}
	}

	ArpMenuField*	field = new ArpMenuField("motion_field", NULL, menu);
	if (!field) {
		delete menu;
		return NULL;
	}
	field->SetDivider(0);
	return field;
}

_VaccineFilterSettings::_VaccineFilterSettings(	AmFilterHolderI* target,
												const BMessage& initSettings)
		: AmFilterConfigLayout(target, initSettings),
		  mFollowXBox(NULL), mInvertXBox(NULL), mRhythmField(NULL),
		  mTrackMotionBox(NULL), mEditor(NULL)
{
	float	labelW = -1, intW = -1;
	const BFont*	font = be_plain_font;
	if (font) {
		labelW = font->StringWidth("Proximity to beat:");
		intW = font->StringWidth("Furthest") + 5;
	}

	try {
		ArpBaseLayout*	topVBar = (new ArpRunningBar("TopVBar"))
										->SetParams(ArpMessage()
											.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
											.SetFloat(ArpRunningBar::IntraSpaceP, .5)
										);
		AddLayoutChild( topVBar );
		topVBar->AddLayoutChild((new ArpTextControl(
									SZ_FILTER_LABEL, "Label:","",
									mImpl.AttachTextControl(SZ_FILTER_LABEL)))
					->SetParams(ArpMessage()
						.SetString(ArpTextControl::MinTextStringP, "8")
						.SetString(ArpTextControl::PrefTextStringP, "8888888888")
					)
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,0)
						.SetInt32(ArpRunningBar::FillC,ArpEastWest)
					)
				);
		ArpBaseLayout*	colHBar = (new ArpRunningBar("ColHBar"))
										->SetParams(ArpMessage()
											.SetInt32(ArpRunningBar::OrientationP, B_HORIZONTAL)
											.SetFloat(ArpRunningBar::IntraSpaceP, .5)
										);
		topVBar->AddLayoutChild( colHBar );
		AddViews(colHBar, mImpl, labelW, intW, initSettings);
	} catch(...) {
		throw;
	}
	Implementation().RefreshControls(mSettings);

	if (mTrackMotionBox && mRhythmField) {
		if (mTrackMotionBox->Value() == B_CONTROL_ON) mRhythmField->SetEnabled(false);
	}
	if (mFollowXBox && mInvertXBox) {
		if (mFollowXBox->Value() != B_CONTROL_ON) mInvertXBox->SetEnabled(false);
	}

}

void _VaccineFilterSettings::AttachedToWindow()
{
	inherited::AttachedToWindow();
	if (mRhythmField) mRhythmField->Menu()->SetTargetForItems(this);
}

void _VaccineFilterSettings::MessageReceived(BMessage* msg)
{
	switch (msg->what) {
		case MOTION_MSG: {
			const char*		key;
			if (msg->FindString(MOTION_KEY_STR, &key) == B_OK) {
				AmMotionI*	m = AmGlobals().NewMotion(key);
				if (m) {
					BMessage		rMsg, containerMsg;
					if (m->WriteTo(rMsg) == B_OK && containerMsg.AddMessage(MOTION_STR, &rMsg) == B_OK) {
						Implementation().SendConfiguration(&containerMsg);
						mSettings.Update(containerMsg);
						if (mEditor) mEditor->Refresh(containerMsg);
					}
					delete m;
				}
			}
		} break;
		case ArpConfigureImpl::CONFIG_REPORT_MSG: {
			const char*		param;
			if (msg->FindString("arp:param", &param) == B_OK) {
				if (mRhythmField && strcmp(param, MOTION_FROM_TRACK_STR) == 0) {
					if (mTrackMotionBox && mTrackMotionBox->Value() == B_CONTROL_ON)
						mRhythmField->SetEnabled(false);
					else mRhythmField->SetEnabled(true);
				} else if (mInvertXBox && strcmp(param, FOLLOW_X_STR) == 0) {
					if (mFollowXBox && mFollowXBox->Value() == B_CONTROL_ON)
						mInvertXBox->SetEnabled(true);
					else mInvertXBox->SetEnabled(false);
				}
			}
		} // Note: no break on purpose				
		default:
			inherited::MessageReceived(msg);
	}
}

void _VaccineFilterSettings::AddViews(	ArpBaseLayout* toLayout, ArpConfigureImpl& impl,
										float labelW, float intW,
										const BMessage& initSettings)
{
	ArpBaseLayout*	vBar = (new ArpRunningBar("SubVBar"))
										->SetParams(ArpMessage()
										.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
										.SetFloat(ArpRunningBar::IntraSpaceP, .5)
									);
	toLayout->AddLayoutChild(vBar);

	ArpBaseLayout*	box = NULL;
	ArpKnobPanel*	kp = NULL;
	vBar->AddLayoutChild((box = new ArpBox("AmountBox", "Amount"))
		->SetConstraints(ArpMessage()
			.SetFloat(ArpRunningBar::WeightC,0)
			.SetInt32(ArpRunningBar::FillC,ArpFillAll)
			.SetBool(ArpRunningBar::AlignLabelsC,false)
		)
	);
	if (box) {
		box->AddLayoutChild((new ArpRunningBar("AmountVBar"))
			->SetParams(ArpMessage()
				.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
				.SetFloat(ArpRunningBar::IntraSpaceP, .5)
			)
			->AddLayoutChild((new ArpViewWrapper(kp = new ArpKnobPanel(AMOUNT_STR, "Amount:", impl.AttachControl(AMOUNT_STR), MIN_AMOUNT, MAX_AMOUNT, true, B_HORIZONTAL, ARP_TIGHT_RING_ADORNMENT, labelW, intW)))
				->SetConstraints(ArpMessage()
					.SetFloat(ArpRunningBar::WeightC,0)
					.SetInt32(ArpRunningBar::FillC,ArpEastWest)
				)
			)
			->AddLayoutChild((new ArpViewWrapper(new BCheckBox( BRect(0,0,0,0),
												FOLLOW_Y_STR, "Tools follow Y",
												mImpl.AttachCheckBox(FOLLOW_Y_STR))))										
				->SetConstraints(ArpMessage()
					.SetFloat(ArpRunningBar::WeightC,0)
					.SetInt32(ArpRunningBar::FillC,ArpEastWest)
				)
			)
			->AddLayoutChild((new ArpViewWrapper(mFollowXBox = new BCheckBox( BRect(0,0,0,0),
												FOLLOW_X_STR, "Tools follow X",
												mImpl.AttachCheckBox(FOLLOW_X_STR))))										
				->SetConstraints(ArpMessage()
					.SetFloat(ArpRunningBar::WeightC,0)
					.SetInt32(ArpRunningBar::FillC,ArpEastWest)
				)
			)
			->AddLayoutChild((new ArpViewWrapper(mInvertXBox = new BCheckBox( BRect(0,0,0,0),
												INVERT_X_STR, "Invert X",
												mImpl.AttachCheckBox(INVERT_X_STR))))										
				->SetConstraints(ArpMessage()
					.SetFloat(ArpRunningBar::WeightC,0)
					.SetInt32(ArpRunningBar::FillC,ArpEastWest)
				)
			)
		);
	}

	/* This should be for the Amount control, no one interfere.
	 */
	if (kp) {
		ArpIntControl*	intCtrl = kp->IntControl();
		if (intCtrl) intCtrl->SetFormatter(new _OffPercentFormat() );
	}


	vBar->AddLayoutChild((new ArpViewWrapper(kp = new ArpKnobPanel(FREQUENCY_STR, "Frequency:", impl.AttachControl(FREQUENCY_STR), 0, 100, true, B_HORIZONTAL, ARP_TIGHT_RING_ADORNMENT, labelW, intW)))
			->SetConstraints(ArpMessage()
				.SetFloat(ArpRunningBar::WeightC,0)
				.SetInt32(ArpRunningBar::FillC,ArpEastWest)));
	if (kp) {
		ArpIntControl*	intCtrl = kp->IntControl();
		if (intCtrl) intCtrl->SetFormatter(arp_new_frequency_formatter() );
	}

	vBar->AddLayoutChild((box = new ArpBox("MotionBox", "Motion Selection"))
		->SetConstraints(ArpMessage()
			.SetFloat(ArpRunningBar::WeightC,0)
			.SetInt32(ArpRunningBar::FillC,ArpFillAll)
			.SetBool(ArpRunningBar::AlignLabelsC,false)
		)
	);
	mRhythmField = new_rhythm_menu_field();
	if (box && mRhythmField) {
		box->AddLayoutChild((new ArpRunningBar("DurationVBar"))
			->SetParams(ArpMessage()
				.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
				.SetFloat(ArpRunningBar::IntraSpaceP, .5)
			)
			->AddLayoutChild(mRhythmField
				->SetConstraints(ArpMessage()
					.SetFloat(ArpRunningBar::WeightC,0)
					.SetInt32(ArpRunningBar::FillC,ArpEastWest)
				)
			)
			->AddLayoutChild((new ArpViewWrapper(mTrackMotionBox = new BCheckBox( BRect(0,0,0,0),
											MOTION_FROM_TRACK_STR, "Use motion from track",
										mImpl.AttachCheckBox(MOTION_FROM_TRACK_STR))))										
				->SetConstraints(ArpMessage()
					.SetFloat(ArpRunningBar::WeightC,0)
					.SetInt32(ArpRunningBar::FillC,ArpEastWest)
				)
			)
		);
	} 
		
	vBar->AddLayoutChild(mEditor = new AmMotionEditor("motion_editor", this, initSettings, MOTION_STR))
			->SetConstraints(ArpMessage()
				.SetFloat(ArpRunningBar::WeightC,1)
				.SetInt32(ArpRunningBar::FillC,ArpEastWest));
}
