#include "AsDefs.h"
#include "AsAnnotater.h"

/*************************************************************************
 * AS-ANNOTATER
 *************************************************************************/
AsAnnotater::AsAnnotater(BMessage* device, const entry_ref* ref)
		: mStatus(B_OK), mRef(ref)
{
	if (device) mDevice = *device;
}

AsAnnotater::~AsAnnotater()
{
	if (mStatus == B_OK && mRef) {
		BFile		file(mRef, B_WRITE_ONLY | B_CREATE_FILE | B_ERASE_FILE);
		if (file.InitCheck() == B_OK) {
			mDevice.Flatten(&file);
		}
	}
}

status_t AsAnnotater::RemoveSysExCommands()
{
	mDevice.RemoveName(AS_SYSEX_COMMAND_NAME);
	return B_OK;
}

status_t AsAnnotater::AddSysExMultiCommand(	vector<AsSystemExclusive*>& sysex,
											int32 initValue, const BString& key,
											const AsCommandValue& deviceId,
											const AsCommandValue& value,
											const AsCommandValue* channel,
											vector<BString>* valueLabels, uint32 bitNum,
											uint32 flags)
{
	BMessage		command;
	status_t		err;
	/* Add in the actual sysex (and delete it)
	 */
	for (uint32 k = 0; k < sysex.size(); k++) {
		if (sysex[k]) {
			if ((err = command.AddData("syx", B_INT8_TYPE, (void*)sysex[k]->Data(), sysex[k]->Size())) != B_OK) return err;
			delete sysex[k];
		}
	}
	sysex.resize(0);
	/* Add the init value and key
	 */
	if ((err = command.AddInt32("iv", initValue)) != B_OK) return err;
	if ((err = command.AddString("key", key)) != B_OK) return err;
	/* Add the various command values.
	 */
	if ((err = AddCommandValue("cv_did", deviceId, command)) != B_OK) return err;
	if ((err = AddCommandValue("cv_v", value, command)) != B_OK) return err;
	if (channel)
		if ((err = AddCommandValue("cv_c", *channel, command)) != B_OK) return err;
	/* Add the value labels.
	 */
	if (valueLabels) {
		for (uint32 k = 0; k < valueLabels->size(); k++) {
			if ((err = command.AddString("vl", (*valueLabels)[k])) != B_OK) return err;
		}
	}
	/* Add the bit num and flags
	 */
	if ((err = command.AddInt32("bn", bitNum)) != B_OK) return err;
	if ((err = command.AddInt32("f", flags)) != B_OK) return err;

	return mDevice.AddMessage(AS_SYSEX_COMMAND_NAME, &command);
}

status_t AsAnnotater::AddCommandValue(	const char* name, const AsCommandValue& cv,
										BMessage& commandMsg)
{
	BMessage		cvMsg;
	status_t		err;
	if ((err = cvMsg.AddInt32("i", cv.index)) != B_OK) return err;
	if ((err = cvMsg.AddInt32("s", cv.start)) != B_OK) return err;
	if ((err = cvMsg.AddInt32("e", cv.end)) != B_OK) return err;
	if ((err = cvMsg.AddInt32("mi", cv.min)) != B_OK) return err;
	if ((err = cvMsg.AddInt32("ma", cv.max)) != B_OK) return err;
	return commandMsg.AddMessage(name, &cvMsg);
}

/*************************************************************************
 * AS-SYSTEM-EXCLUSIVE
 * A simple wrapper around some bytes.
 *************************************************************************/
AsSystemExclusive::AsSystemExclusive(const uint8* data, size_t size)
		: mData(0), mSize(0)
{
	SetTo(data, size);
}

AsSystemExclusive::AsSystemExclusive(const AsSystemExclusive& o)
		: mData(0), mSize(0)
{
	if (o.mData) SetTo(o.mData, o.mSize);
}

AsSystemExclusive::~AsSystemExclusive()
{
	Free();
}

AsSystemExclusive& AsSystemExclusive::operator=(const AsSystemExclusive& o)
{
	mData = o.mData;
	return *this;
}

uint8* AsSystemExclusive::Data()
{
	return mData;
}

size_t AsSystemExclusive::Size()
{
	return mSize;
}

status_t AsSystemExclusive::SetTo(const uint8* data, size_t size)
{
	Free();
	if (!data || size < 1) return B_OK;
	mData = new uint8[size];
	if (!mData) return B_NO_MEMORY;
	mSize = size;
	for (uint32 k = 0; k < size; k++) mData[k] = data[k];
	return B_OK;
}

void AsSystemExclusive::Free()
{
	delete mData;
	mData = 0;
	mSize = 0;
}

/***************************************************************************
 * AS-COMMAND-VALUE
 ***************************************************************************/
AsCommandValue::AsCommandValue()
		: index(0), start(0), end(0), min(0), max(0)
{
}

AsCommandValue::AsCommandValue(	int32 inIndex, uint32 inStart, uint32 inEnd,
								int32 inMin, int32 inMax)
		: index(inIndex), start(inStart), end(inEnd), min(inMin), max(inMax)
{
}
		
AsCommandValue::AsCommandValue(const AsCommandValue& o)
		: index(o.index), start(o.start), end(o.end), min(o.min), max(o.max)
{
}

AsCommandValue& AsCommandValue::operator=(const AsCommandValue& o)
{
	index = o.index;
	start = o.start;
	end = o.end;
	min = o.min;
	max = o.max;
	return *this;
}
